
#ifndef nmbgfiol_included
    #include "nmbgfiol.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <ctype.h>
#include <string.h>
#include <math.h>




/*
    This library is used to write NMBGF version 2.3 files.
*/
#define  MajorNMBGFVersion  2
#define  MinorNMBGFVersion  3




#define  StackMaxHeight  100

#define  BufferNodeDelta  (0x1000 - sizeof (void*))
    /*
        We want BufferNodeStruct to be a power of two in size, to
        make the mallocs more efficient.
    */


struct BufferNodeStruct
{
    char                     Data [BufferNodeDelta];
    struct BufferNodeStruct* Next;
};


struct BufferInfoStruct
{
    NMBGF_INTEGER            BufferLength;
    NMBGF_INTEGER            ClientCounter;
    NMBGF_INTEGER            ElementCounter;
    int                      SuppressAutoClose;
    struct BufferNodeStruct* Head;
    struct BufferNodeStruct* Tail;
};




#define LastElement_OpeningBrace 0
#define LastElement_ClosingBrace 1
#define LastElement_Primitive    2
#define LastElement_EndOfLine    3

struct NMBGF_OutputContext
{
    void*                    FileHandle;
    NMBGF_FileCloseFunType   FileCloseFun;
    NMBGF_FileWriteFunType   FileWriteFun;
    int                      ErrorFlag;
    int                      StackHeight;
    int                      ASCIIMode;
    int                      LastElement;
    int                      IndentLevel;
    char*                    WorkBuffer;
    long int                 WorkBufferLength;
    struct BufferInfoStruct* Stack;
    struct BufferNodeStruct* ReturnedBufferNodesHead;
};








/******************************************************************************/
/*
/*    Miscellaneous Utility Functions
/*
/******************************************************************************/




NMBGF_INTEGER ComputeSizeOfStringInFile (NMBGF_INTEGER NumCharsInString)
/*
  Computes how many bytes in a NMBGF file are required to hold a string
  of NumCharsInString characters in length.  Recall that every item in
  an NMBGF file must be a multiple of 4 bytes in size.  Therefore, if
  a string is not an exact multiple of 4 characters in length, then the
  string must be padded with dummy characters at the end to bring its
  length in the file up to an exact multiple of 4 bytes.
*/
{
    return 4 * ((NumCharsInString + 3) / 4);
};




static char* GetWorkBuffer (NMBGF_HOC hOC, long int MinBufferLength)
{
    if (hOC->WorkBufferLength < MinBufferLength)
    {
        free (hOC->WorkBuffer);

        hOC->WorkBuffer = (char*) malloc (MinBufferLength);

        hOC->WorkBufferLength = MinBufferLength;
    };

    return hOC->WorkBuffer;
};








/******************************************************************************/
/*
/*    Error Reporting Functions
/*
/******************************************************************************/




static void SetErrorFlag (NMBGF_HOC hOC)
{
    hOC->ErrorFlag = 1;
};


int NMBGF_WasError (NMBGF_HOC hOC)
/*
    Returns a non-zero value if an error has occured in this library.
*/
{
    return hOC->ErrorFlag;
};


void NMBGF_ResetErrorFlag (NMBGF_HOC hOC)
/*
    Sets the error flag to its non-error state.
*/
{
    hOC->ErrorFlag = 0;
};








/******************************************************************************/
/*
/*    Implement a Stack of Output Stream Buffers
/*
/******************************************************************************/




struct BufferNodeStruct* GetNewBufferNode (NMBGF_HOC hOC)
{
    struct BufferNodeStruct* BufferNode;

    if (hOC->ReturnedBufferNodesHead)
    {
        BufferNode                   = hOC->ReturnedBufferNodesHead;
        hOC->ReturnedBufferNodesHead = hOC->ReturnedBufferNodesHead->Next;
    }
    else
    {
        BufferNode = (struct BufferNodeStruct*) malloc
            (sizeof (struct BufferNodeStruct));
    };

    BufferNode->Next = NULL;

    return BufferNode;
};




void ReturnBufferNode (NMBGF_HOC hOC, struct BufferNodeStruct* BufferNode)
{
    BufferNode->Next              = hOC->ReturnedBufferNodesHead;
    hOC->ReturnedBufferNodesHead = BufferNode;
};




void FreeBufferNodes (NMBGF_HOC hOC)
{
    struct BufferNodeStruct* NodeToFree;

    while (hOC->ReturnedBufferNodesHead)
    {
        NodeToFree                    = hOC->ReturnedBufferNodesHead;
        hOC->ReturnedBufferNodesHead = hOC->ReturnedBufferNodesHead->Next;

        free (NodeToFree);
    };
};




void InitializeOutputStreamBufferStack (NMBGF_HOC hOC)
{
    hOC->StackHeight             = -1;
    hOC->ReturnedBufferNodesHead = NULL;

    hOC->Stack = (struct BufferInfoStruct*) malloc
        (StackMaxHeight * sizeof (struct BufferInfoStruct));
};




void DeinitializeOutputStreamBufferStack (NMBGF_HOC hOC)
{
    assert (hOC->StackHeight == -1);

    free (hOC->Stack);

    FreeBufferNodes (hOC);
};








/******************************************************************************/
/*
/*    Output Stream Buffer Stack Access Functions
/*
/******************************************************************************/




void NMBGF_PushOutputStreamBuffer (NMBGF_HOC hOC)
/*
    Push a new, initially empty output stream buffer onto the top of
    the stack.  All data sent via the PutToTopOutputStreamBuffer function
    will accumulate in this buffer until the next call to
    PopOutputStreamBuffer.
*/
{
    ++(hOC->StackHeight);

    assert (hOC->StackHeight < StackMaxHeight);
    assert (hOC->FileHandle != NULL);

    hOC->Stack [hOC->StackHeight].BufferLength   = 0;
    hOC->Stack [hOC->StackHeight].ClientCounter  = 0;
    hOC->Stack [hOC->StackHeight].ElementCounter = 0;

    hOC->Stack [hOC->StackHeight].SuppressAutoClose = 0;

    hOC->Stack [hOC->StackHeight].Head =
        hOC->Stack [hOC->StackHeight].Tail =
            GetNewBufferNode (hOC);
};




NMBGF_INTEGER NMBGF_GetTopOutputStreamBufferLengthInBytes (NMBGF_HOC hOC)
/*
    Returns the length, in bytes, of the output stream buffer that is on top
    of the output stream buffer stack.
*/
{
    assert (hOC->StackHeight != -1);

    return hOC->Stack[hOC->StackHeight].BufferLength;
};




void NMBGF_SetTopClientStackCounter (NMBGF_HOC hOC, NMBGF_INTEGER X)
/*
    Sets the value of the client counter associated with the topmost
    output stream buffer on the output stream buffer stack.

    The client counter is provided for the convenience of the caller of
    the output stream buffer stack access functions.  It can be used to
    maintain a user-defined value which is associated with each stream
    buffer on the stack.
*/
{
    assert (hOC->StackHeight != -1);

    hOC->Stack[hOC->StackHeight].ClientCounter = X;
};


void NMBGF_IncrementTopClientStackCounter (NMBGF_HOC hOC)
/*
    Increments the value of the client counter associated with the
    topmost output stream buffer on the output stream buffer stack.
*/
{
    assert (hOC->StackHeight != -1);

    ++(hOC->Stack[hOC->StackHeight].ClientCounter);
};


NMBGF_INTEGER NMBGF_GetTopClientStackCounter (NMBGF_HOC hOC)
/*
    Returns the value of the client counter associated with the topmost
    output stream buffer on the output stream buffer stack.
*/
{
    assert (hOC->StackHeight != -1);

    return hOC->Stack[hOC->StackHeight].ClientCounter;
};




static void IncrementTopStackElementCounter (NMBGF_HOC hOC)
/*
    Increments the value of the element counter associated with the
    topmost output stream buffer on the output stream buffer stack.
    This is done whenever a primative data elements (ie, one of the
    five basic NMBGF datatypes) is written.
*/
{
    assert (hOC->StackHeight != -1);

    ++(hOC->Stack[hOC->StackHeight].ElementCounter);
};


NMBGF_INTEGER NMBGF_GetTopStackElementCounter (NMBGF_HOC hOC)
/*
    Returns the value of the element counter associated with the topmost
    output stream buffer on the output stream buffer stack.  This is the
    number of primative data elements (ie, one of the five basic NMBGF
    datatypes) that have been written to this output stream buffer.
*/
{
    assert (hOC->StackHeight != -1);

    return hOC->Stack[hOC->StackHeight].ElementCounter;
};




void NMBGF_PutToTopOutputStreamBuffer
    (NMBGF_HOC hOC, void* DataBlock, NMBGF_INTEGER DataBlockLength)
/*
    Appends the data pointed to by DataBlock to the end of the output stream
    buffer that is at the top of the output stream buffer stack.  If there
    is no output stream buffer on the output stream buffer stack, then the
    data is written to file hOC->FileHandle.
*/
{
    NMBGF_INTEGER SpaceRemainingInCurrentNode;
    NMBGF_INTEGER SpaceUsedInCurrentNode;
    NMBGF_INTEGER NumBytesToCopy;


    if (hOC->StackHeight == -1)
    {
        if (hOC->FileWriteFun (hOC->FileHandle, DataBlock, DataBlockLength))
            SetErrorFlag (hOC);

        return;
    };

    while (DataBlockLength > 0)
    {
        SpaceUsedInCurrentNode =
            hOC->Stack [hOC->StackHeight].BufferLength % BufferNodeDelta;

        if ((SpaceUsedInCurrentNode           == 0) &&
            (hOC->Stack [hOC->StackHeight].BufferLength != 0))
            SpaceUsedInCurrentNode = BufferNodeDelta;

        SpaceRemainingInCurrentNode = BufferNodeDelta - SpaceUsedInCurrentNode;

        if (SpaceRemainingInCurrentNode == 0)
        {
            hOC->Stack [hOC->StackHeight].Tail->Next = GetNewBufferNode (hOC);

            hOC->Stack [hOC->StackHeight].Tail =
                hOC->Stack [hOC->StackHeight].Tail->Next;

            SpaceUsedInCurrentNode         = 0;
            SpaceRemainingInCurrentNode    = BufferNodeDelta;
        };

        NumBytesToCopy = (DataBlockLength < SpaceRemainingInCurrentNode) ?
            DataBlockLength : SpaceRemainingInCurrentNode;

        memcpy (&hOC->Stack [hOC->StackHeight].Tail->Data
                    [(size_t)SpaceUsedInCurrentNode],
                DataBlock,
                (size_t) NumBytesToCopy);

        hOC->Stack [hOC->StackHeight].BufferLength += NumBytesToCopy;

        DataBlock = (char*) DataBlock + (size_t) NumBytesToCopy;

        DataBlockLength -= NumBytesToCopy;
    };

    assert (DataBlockLength == 0);
};








static void PutToTopOutputStreamBuffer_IndentSpaces (NMBGF_HOC hOC)
{
    int i;
    int n = hOC->IndentLevel;

    for (i = 0; i < n; ++i)
        NMBGF_PutToTopOutputStreamBuffer (hOC, "    ", 4);
        /*  Indent 4 spaces for each level  */
};




void NMBGF_PutToTopOutputStreamBuffer_EndOfLine (NMBGF_HOC hOC)
{
    assert (hOC->ASCIIMode);

    NMBGF_PutToTopOutputStreamBuffer (hOC, "\r\n", 2);

    hOC->LastElement = LastElement_EndOfLine;
};


static void PutToTopOutputStreamBuffer_OpeningBrace (NMBGF_HOC hOC)
{
    assert (hOC->ASCIIMode);

    if (hOC->LastElement != LastElement_EndOfLine)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    PutToTopOutputStreamBuffer_IndentSpaces (hOC);

    NMBGF_PutToTopOutputStreamBuffer (hOC, "{", 1);

    hOC->LastElement = LastElement_OpeningBrace;

    ++(hOC->IndentLevel);
};


static void PutToTopOutputStreamBuffer_ClosingBrace (NMBGF_HOC hOC)
{
    assert (hOC->ASCIIMode);

    --(hOC->IndentLevel);

    if (hOC->LastElement == LastElement_EndOfLine)
        PutToTopOutputStreamBuffer_IndentSpaces (hOC);

    NMBGF_PutToTopOutputStreamBuffer (hOC, "}", 1);

    hOC->LastElement = LastElement_ClosingBrace;
};


static void PutToTopOutputStreamBuffer_Primitive
    (NMBGF_HOC hOC, void* DataBlock, NMBGF_INTEGER DataBlockLength)
{
    assert (hOC->ASCIIMode);

    switch (hOC->LastElement)
    {
    case LastElement_Primitive:
        NMBGF_PutToTopOutputStreamBuffer (hOC, " ", 1);
        break;
    case LastElement_EndOfLine:
        PutToTopOutputStreamBuffer_IndentSpaces (hOC);
        break;
    };

    NMBGF_PutToTopOutputStreamBuffer (hOC, DataBlock, DataBlockLength);

    hOC->LastElement = LastElement_Primitive;
};








#define  ASCIIBufferSize  128

static char ASCIIBuffer [ASCIIBufferSize];

static int ASCIIBufferDataLength = 0;


static void PutLiteralIntoASCIIBuffer (NMBGF_LITERAL X)
{
    ASCIIBuffer[0] = X [0];
    ASCIIBuffer[1] = X [1];
    ASCIIBuffer[2] = X [2];
    ASCIIBuffer[3] = X [3];

    ASCIIBufferDataLength = 4;
};


static void PutIntegerIntoASCIIBuffer (NMBGF_INTEGER X)
{
    ASCIIBufferDataLength = sprintf (ASCIIBuffer, "%ld", X);
};

static void PutFloatIntoASCIIBuffer (NMBGF_FLOAT X)
{
    double d;

    d = X;

    ASCIIBufferDataLength = sprintf (ASCIIBuffer, "%.8g", d);
};

static void PutCoordinateIntoASCIIBuffer (NMBGF_COORDINATE X)
{
    double d1;
    double d2;

    d1 = X.C1;
    d2 = X.C2;

    ASCIIBufferDataLength = sprintf (ASCIIBuffer, "(%.10g,%.10g)", d1, d2);
};








void NMBGF_WriteIntegerToPreceedingBuffer (NMBGF_HOC hOC, NMBGF_INTEGER X)
/*
    Appends the integer X to the end of the preceeding output stream
    buffer (ie, the buffer that would be on top of the output stream buffer
    stack if it were to be popped).  If there is currently just one output
    stream buffer on the output stream buffer stack, then write the integer
    X to the output file.
*/
{
    assert (hOC->StackHeight > -1);

    --(hOC->StackHeight);

    if (hOC->ASCIIMode)
    {
        NMBGF_PutToTopOutputStreamBuffer (hOC, " ", 1);

        PutIntegerIntoASCIIBuffer (X);

        NMBGF_PutToTopOutputStreamBuffer
            (hOC, &ASCIIBuffer, ASCIIBufferDataLength);

        NMBGF_PutToTopOutputStreamBuffer (hOC, " ", 1);
    }
    else
        NMBGF_PutToTopOutputStreamBuffer (hOC, &X, sizeof (NMBGF_INTEGER));

    ++(hOC->StackHeight);
};




void NMBGF_PopOutputStreamBuffer (NMBGF_HOC hOC)
/*
    1. Pop the current output stream buffer from the output stream buffer stack.
    2. If the stack is empty, write the data from the popped buffer to the
       output file fot.  If the stack is not empty, then append the data
       from the popped buffer to the end of the new topmost output stream
       buffer.
    3. Destroy the buffer.
*/
{
    struct BufferNodeStruct* BufferNode;
    struct BufferNodeStruct* BufferNodeToReturn;
    NMBGF_INTEGER            n;
    NMBGF_INTEGER            m;


    assert (hOC->StackHeight > -1);
    assert (hOC->FileHandle != NULL);


    BufferNode = hOC->Stack [hOC->StackHeight].Head;
    n          = hOC->Stack [hOC->StackHeight].BufferLength;

    --(hOC->StackHeight);


    while (BufferNode)
    {
        assert (BufferNode);

        if (n != 0)
        {
            m = (n < (NMBGF_INTEGER)BufferNodeDelta) ? n : BufferNodeDelta;

            assert (hOC->ASCIIMode  |  (((m/4)*4) == m));
                /*  If in binary mode, m must be a multiple of four  */

            if (hOC->StackHeight == -1)
            {
                if (hOC->FileWriteFun (hOC->FileHandle, BufferNode->Data, m))
                    SetErrorFlag (hOC);
            }
            else
            {
                NMBGF_PutToTopOutputStreamBuffer (hOC, BufferNode->Data, m);
            };

            n -= m;
        };

        BufferNodeToReturn = BufferNode;

        BufferNode = BufferNode->Next;

        ReturnBufferNode (hOC, BufferNodeToReturn);
    };

    assert (n == 0);
};




void NMBGF_AutoClosePreviousSection (NMBGF_HOC hOC)
/*
    This function should be called just before a new NMBGF section is
    begun.  Unless autoclose is disabled, it closes the previous section
    by calculating its length, then writing its length and its contents
    to the next lower output stream buffer on the output stream buffer stack.
*/
{
    NMBGF_INTEGER LengthOfSectionBeingClosed;


    assert (hOC->StackHeight > -1);


    if (hOC->Stack [hOC->StackHeight].SuppressAutoClose)
        hOC->Stack [hOC->StackHeight].SuppressAutoClose = 0;
    else
    {
        if (hOC->ASCIIMode)
        {
            PutToTopOutputStreamBuffer_ClosingBrace (hOC);
        }
        else
        {
            LengthOfSectionBeingClosed =
                NMBGF_GetTopOutputStreamBufferLengthInBytes(hOC) / 4;

            NMBGF_WriteIntegerToPreceedingBuffer (hOC, LengthOfSectionBeingClosed);
        };

        NMBGF_PopOutputStreamBuffer (hOC);
    };
};




void NMBGF_BeginOptionalSubsections (NMBGF_HOC hOC)
/*
    This function should only be called directly after all required parameters
    for a NMBGF section have been written.  It indicates that the section
    currently being written has one or more optional subsections.  All
    sections written between a call to this function and the matching call
    to function NMBGF_EndOptionalSubsections are taken to be subsections of
    the section that was being written when this function was called.
*/
{
    assert (hOC->StackHeight > -1);

    hOC->Stack [hOC->StackHeight].SuppressAutoClose = 1;
};




void NMBGF_EndOptionalSubsections (NMBGF_HOC hOC)
/*
    See comments for function NMBGF_BeginOptionalSubsections.
*/
{
    NMBGF_AutoClosePreviousSection (hOC);
};








/******************************************************************************/
/*
/*    Function to Encode ASCII versions of strings
/*
/******************************************************************************/




/*
/*  Rules for Encoding Strings in NMAGF ASCII grid files.
/*
/*  1. String is delimited by double quotes "".
/*  2. All characters are literals except...
/*      Embedded double quotes are encoded by the 3-letter code {"}.
/*      Embedded opening curley brackets are encoded by the 3-letter code {{}.
/*      Embedded closing curley brackets are encoded by the 3-letter code {}}.
/*      Newlines (ASCII 10) are encoded by the 3-letter code {n}.
/*      Carriage (ASCII 13) returns are encoded by the 3-letter code {r}.
/*      Any other character may be encoded by the 4-letter code
/*          {##}, where ## represents two hexidecimal digits (0-9, A-F, a-f)
/*          specifying the ASCII value of the character..
*/




static NMBGF_INTEGER EncodeASCIIString
    (const char* Src, NMBGF_INTEGER SrcLength, char* Dest)
/*
/*  Takes the character string Src, of length SrcLength, and encodes
/*  it so that it can be written to an ASCII NMAGF file.  The length
/*  of Dest should be at least 4*SrcLength + 2.
*/
{
    char HexCodes [] = "0123456789ABCDEF";

    NMBGF_INTEGER i;
    char          C;

    char* OrigDest = Dest;

    (*Dest++) = '"';    /*  Opening delimiting quote  */

    for (i = 0; i < SrcLength; ++i)
    {
        C = Src [i];

        switch (C)
        {
        case '"':
        case '{':
        case '}':
            (*Dest++) = '{';
            (*Dest++) = C;
            (*Dest++) = '}';
            break;
        case '\n':
            (*Dest++) = '{';
            (*Dest++) = 'n';
            (*Dest++) = '}';
            break;
        case '\r':
            (*Dest++) = '{';
            (*Dest++) = 'r';
            (*Dest++) = '}';
            break;
        default:
            if (iscntrl (C))
            {
                (*Dest++) = '{';
                (*Dest++) = HexCodes [(((unsigned char)(C)) & 0xF0) >> 4];
                (*Dest++) = HexCodes [((unsigned char)(C)) & 0x0F];
                (*Dest++) = '}';
            }
            else
                (*Dest++) = C;
        };
    };

    (*Dest++) = '"';    /*  Closing delimiting quote  */

    return (Dest - OrigDest);
};








/******************************************************************************/
/*
/*    Functions to Write the Core NMBGF Data Types
/*
/******************************************************************************/




void NMBGF_PutLiteral (NMBGF_HOC hOC, NMBGF_LITERAL X)
/*
    Writes a value of type NMBGF_LITERAL to the topmost output stream
    buffer on the output stream buffer stack.
*/
{
    assert (hOC->FileHandle);

    if (hOC->ASCIIMode)
    {
        PutLiteralIntoASCIIBuffer (X);

        PutToTopOutputStreamBuffer_Primitive
            (hOC, &ASCIIBuffer, ASCIIBufferDataLength);
    }
    else
        NMBGF_PutToTopOutputStreamBuffer (hOC, X, sizeof (NMBGF_LITERAL));

    if (hOC->StackHeight >= 0)
        IncrementTopStackElementCounter (hOC);
};




void NMBGF_PutString (NMBGF_HOC hOC, NMBGF_STRING X)
/*
    Writes a value of type NMBGF_STRING to the topmost output stream
    buffer on the output stream buffer stack.  This function handles the
    writing of the strings length, as well as the padding of the string so
    that it occupies a multiple of four bytes.
*/
{
    static char PaddingChars [] = "   ";

    NMBGF_INTEGER Length;
    NMBGF_INTEGER PaddingCharsNeeded;

    char* Buffer;
    int   NewLength;

    assert (hOC->FileHandle);

    Length = (NMBGF_INTEGER) strlen (X);

    if (hOC->ASCIIMode)
    {
        Buffer = GetWorkBuffer (hOC, 4*Length + 2);

        NewLength = EncodeASCIIString (X, Length, Buffer);

        PutToTopOutputStreamBuffer_Primitive (hOC, Buffer, NewLength);
    }
    else
    {
        NMBGF_PutToTopOutputStreamBuffer (hOC, &Length, sizeof (NMBGF_INTEGER));

        NMBGF_PutToTopOutputStreamBuffer (hOC, (char*) X, Length);

        PaddingCharsNeeded = ComputeSizeOfStringInFile (Length) - Length;

        assert (PaddingCharsNeeded >= 0  &&  PaddingCharsNeeded < 4);

        if (PaddingCharsNeeded)
            NMBGF_PutToTopOutputStreamBuffer
                (hOC, PaddingChars, PaddingCharsNeeded);
    };

    if (hOC->StackHeight >= 0)
        IncrementTopStackElementCounter (hOC);
};




void NMBGF_PutInteger (NMBGF_HOC hOC, NMBGF_INTEGER X)
/*
    Writes a value of type NMBGF_INTEGER to the topmost output stream
    buffer on the output stream buffer stack.
*/
{
    assert (hOC->FileHandle);

    if (hOC->ASCIIMode)
    {
        PutIntegerIntoASCIIBuffer (X);

        PutToTopOutputStreamBuffer_Primitive
            (hOC, &ASCIIBuffer, ASCIIBufferDataLength);
    }
    else
        NMBGF_PutToTopOutputStreamBuffer (hOC, &X, sizeof (NMBGF_INTEGER));

    if (hOC->StackHeight >= 0)
        IncrementTopStackElementCounter (hOC);
};




void NMBGF_PutFloat (NMBGF_HOC hOC, NMBGF_FLOAT X)
/*
    Writes a value of type NMBGF_FLOAT to the topmost output stream
    buffer on the output stream buffer stack.
*/
{
    assert (hOC->FileHandle);

    if (hOC->ASCIIMode)
    {
        PutFloatIntoASCIIBuffer (X);

        PutToTopOutputStreamBuffer_Primitive
            (hOC, &ASCIIBuffer, ASCIIBufferDataLength);
    }
    else
        NMBGF_PutToTopOutputStreamBuffer (hOC, &X, sizeof (NMBGF_FLOAT));

    if (hOC->StackHeight >= 0)
        IncrementTopStackElementCounter (hOC);
};




void NMBGF_PutCoordinate (NMBGF_HOC hOC, NMBGF_COORDINATE X)
/*
    Writes a value of type NMBGF_COORDINATE to the topmost output stream
    buffer on the output stream buffer stack.
*/
{
    assert (hOC->FileHandle);

    if (hOC->ASCIIMode)
    {
        PutCoordinateIntoASCIIBuffer (X);

        PutToTopOutputStreamBuffer_Primitive
            (hOC, &ASCIIBuffer, ASCIIBufferDataLength);
    }
    else
    {
        NMBGF_PutToTopOutputStreamBuffer (hOC, &X.C1, sizeof (X.C1));
        NMBGF_PutToTopOutputStreamBuffer (hOC, &X.C2, sizeof (X.C2));
    };

    if (hOC->StackHeight >= 0)
        IncrementTopStackElementCounter (hOC);
};








/******************************************************************************/
/*
/*    Functions to Write NMBGF Sections
/*
/******************************************************************************/




void NMBGF_PutStandardSectionProlog (NMBGF_HOC hOC, NMBGF_LITERAL SectionName)
{
    NMBGF_AutoClosePreviousSection (hOC);

    if (hOC->ASCIIMode)
        PutToTopOutputStreamBuffer_OpeningBrace (hOC);

    NMBGF_PutLiteral (hOC, SectionName);

    NMBGF_PushOutputStreamBuffer (hOC);
};








void NMBGF_PutAREMSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  CATAGORY)
/*
    Writes a AREM section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.

    Instructions for writing an AREM section.
        1. Call NMBGF_PutAREMSection
        2. For every closed polygon in the geographic feature...
            2.1 Call NMBGF_AREM_BeginPoints.
            2.2 Call NMBGF_AREM_PutPoint for every point in the polygon.
            2.3 Call NMBGF_AREM_PointsDone.
        3. Call NMBGF_AREM_ClosedPolygonsDone.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "AREM");

    NMBGF_PutString (hOC, CATAGORY);

    NMBGF_PushOutputStreamBuffer (hOC);

    NMBGF_SetTopClientStackCounter (hOC, 0);
};


void NMBGF_AREM_BeginPoints
    (NMBGF_HOC  hOC)
{
    NMBGF_PushOutputStreamBuffer (hOC);
};


void NMBGF_AREM_PutPoint
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  P)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutCoordinate (hOC, P);
};


void NMBGF_AREM_PointsDone
    (NMBGF_HOC hOC)
{
    assert (hOC->StackHeight > -1);


    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC,
         NMBGF_GetTopStackElementCounter (hOC));

    NMBGF_PopOutputStreamBuffer (hOC);

    NMBGF_IncrementTopClientStackCounter (hOC);
};


void NMBGF_AREM_ClosedPolygonsDone
    (NMBGF_HOC hOC)
{
    assert (hOC->StackHeight > -1);


    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC, NMBGF_GetTopClientStackCounter (hOC));

    NMBGF_PopOutputStreamBuffer (hOC);
};








void NMBGF_PutARESSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  CATAGORY)
/*
    Writes a ARES section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.

    Instructions for writing an ARES section.
        1. Call NMBGF_PutARESSection
        2. Call NMBGF_ARES_PutPoint for every point in the area boundary.
        3. Call NMBGF_ARES_PointsDone.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "ARES");

    NMBGF_PutString (hOC, CATAGORY);

    NMBGF_PushOutputStreamBuffer (hOC);
};


void NMBGF_ARES_PutPoint
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  P)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutCoordinate (hOC, P);
};


void NMBGF_ARES_PointsDone
    (NMBGF_HOC hOC)
{
    assert (hOC->StackHeight > -1);


    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC,
         NMBGF_GetTopStackElementCounter (hOC));

    NMBGF_PopOutputStreamBuffer (hOC);
};








void NMBGF_PutATRCSection
    (NMBGF_HOC         hOC,
     NMBGF_STRING      NAME,
     NMBGF_COORDINATE  VALUE)
/*
    Writes a ATRC section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "ATRC");

    NMBGF_PutString     (hOC, NAME);
    NMBGF_PutCoordinate (hOC, VALUE);
};








void NMBGF_PutATRFSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  NAME,
     NMBGF_FLOAT   VALUE)
/*
    Writes a ATRF section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "ATRF");

    NMBGF_PutString (hOC, NAME);
    NMBGF_PutFloat  (hOC, VALUE);
};








void NMBGF_PutATRISection
    (NMBGF_HOC      hOC,
     NMBGF_STRING   NAME,
     NMBGF_INTEGER  VALUE)
/*
    Writes a ATRI section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "ATRI");

    NMBGF_PutString  (hOC, NAME);
    NMBGF_PutInteger (hOC, VALUE);
};








void NMBGF_PutATRSSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  NAME,
     NMBGF_STRING  VALUE)
/*
    Writes a ATRS section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "ATRS");

    NMBGF_PutString (hOC, NAME);
    NMBGF_PutString (hOC, VALUE);
};








void NMBGF_PutATRTSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  NAME)
/*
    Writes a ATRT section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.

    Instructions for writing an ATRT section.
        1. Call NMBGF_PutATRTSection
        2. For every table column...
            2.1 Call NMBGF_ATRT_AddColumn
        3. Call NMBGF_ATRT_ColumnsDone
        4. For each table row...
            4.1 For each column...
                4.1.1 Call NMBGF_ATRT_PutString, NMBGF_ATRT_PutInteger,
                      NMBGF_ATRT_PutFloat, or NMBGF_ATRT_PutCoordinate to
                      put the data item for this column and row.
            4.2 Call NMBGF_ATRT_RowDone
        5. Call NMBGF_ATRT_TableDone
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "ATRT");

    NMBGF_PutString (hOC, NAME);

    NMBGF_PushOutputStreamBuffer (hOC);

    NMBGF_SetTopClientStackCounter (hOC, 0);
};


void NMBGF_ATRT_AddColumn
    (NMBGF_HOC                hOC,
     NMBGF_STRING             COLNAME,
     NMBGF_ATRT_COLTYPE_ENUM  COLTYPE)
{
    NMBGF_LITERAL ColTypeLiterals [4] = {"STRN", "INTG", "FLOT", "CORD"};

    NMBGF_PutString  (hOC, COLNAME);
    NMBGF_PutLiteral (hOC, ColTypeLiterals [COLTYPE]);

    NMBGF_IncrementTopClientStackCounter (hOC);
};


void NMBGF_ATRT_ColumnsDone
    (NMBGF_HOC  hOC)
{
    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC, NMBGF_GetTopClientStackCounter (hOC));

    NMBGF_PopOutputStreamBuffer (hOC);

    NMBGF_PushOutputStreamBuffer (hOC);

    NMBGF_SetTopClientStackCounter (hOC, 0);

    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);
};


void NMBGF_ATRT_PutString
    (NMBGF_HOC     hOC,
     NMBGF_STRING  X)
{
    NMBGF_PutString (hOC, X);
};


void NMBGF_ATRT_PutInteger
    (NMBGF_HOC      hOC,
     NMBGF_INTEGER  X)
{
    NMBGF_PutInteger (hOC, X);
};


void NMBGF_ATRT_PutFloat
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  X)
{
    NMBGF_PutFloat (hOC, X);
};


void NMBGF_ATRT_PutCoordinate
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  X)
{
    NMBGF_PutCoordinate (hOC, X);
};


void NMBGF_ATRT_RowDone
    (NMBGF_HOC  hOC)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_IncrementTopClientStackCounter (hOC);
};


void NMBGF_ATRT_TableDone
    (NMBGF_HOC  hOC)
{
    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC, NMBGF_GetTopClientStackCounter (hOC));

    NMBGF_PopOutputStreamBuffer (hOC);
};








void NMBGF_PutBKMPSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  FILENAME,
     NMBGF_STRING  CATEGORY)
/*
    Writes a BKMP section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "BKMP");

    NMBGF_PutString (hOC, FILENAME);
    NMBGF_PutString (hOC, CATEGORY);
};








void NMBGF_PutCARTSection
    (NMBGF_HOC             hOC,
     NMBGF_FLOAT           LOR,
     NMBGF_FLOAT           LAR,
     NMBGF_FLOAT           XR,
     NMBGF_FLOAT           YR,
     NMBGF_CART_UNIT_ENUM  UNIT,
     NMBGF_FLOAT           ROT)
/*
    Writes a CART section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_LITERAL UnitLiterals [2] = {"FEET", "METR"};

    NMBGF_PutStandardSectionProlog (hOC, "CART");

    NMBGF_PutFloat   (hOC, LOR);
    NMBGF_PutFloat   (hOC, LAR);
    NMBGF_PutFloat   (hOC, XR);
    NMBGF_PutFloat   (hOC, YR);
    NMBGF_PutLiteral (hOC, UnitLiterals [UNIT]);
    NMBGF_PutFloat   (hOC, ROT);
};








void NMBGF_PutDAPYSection
    (NMBGF_HOC  hOC)
/*
    Writes a DAPY section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.

    Instructions for writing an DAPY section.
        1. Call NMBGF_PutDAPYSection
        2. For every closed polygon in the defined area...
            2.1 Call NMBGF_DAPY_BeginPoints.
            2.2 Call NMBGF_DAPY_PutPoint for every point in the polygon.
            2.3 Call NMBGF_DAPY_PointsDone.
        3. Call NMBGF_DAPY_ClosedPolygonsDone.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "DAPY");

    NMBGF_PushOutputStreamBuffer (hOC);

    NMBGF_SetTopClientStackCounter (hOC, 0);
};


void NMBGF_DAPY_BeginPoints
    (NMBGF_HOC  hOC)
{
    NMBGF_PushOutputStreamBuffer (hOC);
};


void NMBGF_DAPY_PutPoint
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  P)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutCoordinate (hOC, P);
};


void NMBGF_DAPY_PointsDone
    (NMBGF_HOC hOC)
{
    assert (hOC->StackHeight > -1);


    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC,
         NMBGF_GetTopStackElementCounter (hOC));

    NMBGF_PopOutputStreamBuffer (hOC);

    NMBGF_IncrementTopClientStackCounter (hOC);
};


void NMBGF_DAPY_ClosedPolygonsDone
    (NMBGF_HOC hOC)
{
    assert (hOC->StackHeight > -1);


    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC, NMBGF_GetTopClientStackCounter (hOC));

    NMBGF_PopOutputStreamBuffer (hOC);
};








void NMBGF_PutDATESection
    (NMBGF_HOC      hOC,
     NMBGF_INTEGER  DAY,
     NMBGF_INTEGER  MONTH,
     NMBGF_INTEGER  YEAR)
/*
    Writes a DATE section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "DATE");

    NMBGF_PutInteger (hOC, DAY);
    NMBGF_PutInteger (hOC, MONTH);
    NMBGF_PutInteger (hOC, YEAR);
};








void NMBGF_PutDESLSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  D)
/*
    Writes a DESL section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "DESL");

    NMBGF_PutString (hOC, D);
};








void NMBGF_PutDESSSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  D)
/*
    Writes a DESS section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "DESS");

    NMBGF_PutString (hOC, D);
};








void NMBGF_PutENDFSection
    (NMBGF_HOC hOC)
/*
    Writes an ENDF section.  This function is not called directly by the
    user.  Rather, it is automatically called by the function
    NMBGF_EndWritingOutputFile.
*/
{
    NMBGF_AutoClosePreviousSection (hOC);

    if (hOC->ASCIIMode)
    {
        PutToTopOutputStreamBuffer_OpeningBrace (hOC);
        NMBGF_PutLiteral (hOC, "ENDF");
        PutToTopOutputStreamBuffer_ClosingBrace (hOC);
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);
    }
    else
    {
        NMBGF_PutLiteral (hOC, "ENDF");

        NMBGF_PutInteger (hOC, 0);
    };
};








void NMBGF_PutDPALSection
    (NMBGF_HOC  hOC)
/*
    Writes a DPAL section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.

    Instructions for writing an DPAL section.
        1. Call NMBGF_PutDPALSection
        2. Call NMBGF_DPAL_PutDataPoint for every data point.
        3. Call NMBGF_DPAL_DataPointsDone.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "DPAL");

    NMBGF_PushOutputStreamBuffer (hOC);
};


void NMBGF_DPAL_PutDataPoint
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  P,
     NMBGF_FLOAT       V)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutCoordinate (hOC, P);
    NMBGF_PutFloat      (hOC, V);
};


void NMBGF_DPAL_DataPointsDone
    (NMBGF_HOC  hOC)
{
    assert (hOC->StackHeight > -1);


    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC,
         NMBGF_GetTopStackElementCounter (hOC) / 2);

    NMBGF_PopOutputStreamBuffer (hOC);
};








void NMBGF_PutGRIDSection
    (NMBGF_HOC             hOC,
     NMBGF_STRING          NAME,
     NMBGF_INTEGER         NI,
     NMBGF_INTEGER         NJ,
     NMBGF_FLOAT           DI,
     NMBGF_FLOAT           DJ,
     NMBGF_GRID_UNIT_ENUM  UNIT,
     NMBGF_COORDINATE      ORIGIN,
     NMBGF_FLOAT           ROT)
/*
    Writes a GRID section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.

    Instructions for writing a GRID section.
        1. Call NMBGF_PutGRIDSection
        2. For every grid point in the grid...
            2.1 Call NMBGF_GRID_PutGridPoint

    Note that the grid points should be written in the following order:

        for (i = 0; i < NI; ++i)
            for (j = 0; j < NJ; ++j)
                NMBGF_GRID_PutGridPoint (V [i][j]);
*/
{
    NMBGF_LITERAL UnitLiterals [2] = {"FEET", "METR"};

    NMBGF_PutStandardSectionProlog (hOC, "GRID");

    NMBGF_PutString     (hOC, NAME);
    NMBGF_PutInteger    (hOC, NI);
    NMBGF_PutInteger    (hOC, NJ);
    NMBGF_PutFloat      (hOC, DI);
    NMBGF_PutFloat      (hOC, DJ);
    NMBGF_PutLiteral    (hOC, UnitLiterals [UNIT]);
    NMBGF_PutCoordinate (hOC, ORIGIN);
    NMBGF_PutFloat      (hOC, ROT);
};


void NMBGF_GRID_PutGridPoint
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  V)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutFloat (hOC, V);
};








void NMBGF_PutGTSHSection
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  VMIN,
     NMBGF_FLOAT  VMAX)
/*
    Writes a GTSH section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "GTSH");

    NMBGF_PutFloat (hOC, VMIN);
    NMBGF_PutFloat (hOC, VMAX);
};








void NMBGF_PutHEADSection
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  H)
/*
    Writes a HEAD section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "HEAD");

    NMBGF_PutFloat (hOC, H);
};








void NMBGF_PutLINCSection
    (NMBGF_HOC             hOC,
     NMBGF_STRING          CATAGORY,
     NMBGF_LINC_UNIT_ENUM  UNIT,
     NMBGF_COORDINATE      P0,
     NMBGF_FLOAT           H0)
/*
    Writes a LINC section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.

    Instructions for writing a LINC section.
        1. Call NMBGF_PutLINCSection
        2. For each command...
            Call the NMBGF_LINC_Put####Command function corresponding to
            the desired command.
        3. Call NMBGF_LINC_CommandsDone.
*/
{
    NMBGF_LITERAL UnitLiterals [2] = {"FEET", "METR"};

    NMBGF_PutStandardSectionProlog (hOC, "LINC");

    NMBGF_PutString     (hOC, CATAGORY);
    NMBGF_PutLiteral    (hOC, UnitLiterals [UNIT]);
    NMBGF_PutCoordinate (hOC, P0);
    NMBGF_PutFloat      (hOC, H0);

    NMBGF_PushOutputStreamBuffer (hOC);

    NMBGF_SetTopClientStackCounter (hOC, 0);
};


void NMBGF_LINC_PutDSTRCommand
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  DIST)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "DSTR");
    NMBGF_PutFloat   (hOC, DIST);

    NMBGF_IncrementTopClientStackCounter (hOC);
};


void NMBGF_LINC_PutDARRCommand
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  ANGLE,
     NMBGF_FLOAT  RADIUS)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "DARR");
    NMBGF_PutFloat   (hOC, ANGLE);
    NMBGF_PutFloat   (hOC, RADIUS);

    NMBGF_IncrementTopClientStackCounter (hOC);
};


void NMBGF_LINC_PutDALACommand
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  HEAD,
     NMBGF_FLOAT  RADIUS)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "DALA");
    NMBGF_PutFloat   (hOC, HEAD);
    NMBGF_PutFloat   (hOC, RADIUS);

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutDARACommand
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  HEAD,
     NMBGF_FLOAT  RADIUS)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "DARA");
    NMBGF_PutFloat   (hOC, HEAD);
    NMBGF_PutFloat   (hOC, RADIUS);

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutDAPTCommand
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  P)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral    (hOC, "DAPT");
    NMBGF_PutCoordinate (hOC, P);

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutDAPHCommand
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  P,
     NMBGF_FLOAT       HEAD)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral    (hOC, "DAPH");
    NMBGF_PutCoordinate (hOC, P);
    NMBGF_PutFloat      (hOC, HEAD);

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutMSTRCommand
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  DIST)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "MSTR");
    NMBGF_PutFloat   (hOC, DIST);

    NMBGF_IncrementTopClientStackCounter (hOC);
};


void NMBGF_LINC_PutMARRCommand
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  ANGLE,
     NMBGF_FLOAT  RADIUS)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "MARR");
    NMBGF_PutFloat   (hOC, ANGLE);
    NMBGF_PutFloat   (hOC, RADIUS);

    NMBGF_IncrementTopClientStackCounter (hOC);
};


void NMBGF_LINC_PutMALACommand
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  HEAD,
     NMBGF_FLOAT  RADIUS)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "MALA");
    NMBGF_PutFloat   (hOC, HEAD);
    NMBGF_PutFloat   (hOC, RADIUS);

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutMARACommand
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  HEAD,
     NMBGF_FLOAT  RADIUS)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "MARA");
    NMBGF_PutFloat   (hOC, HEAD);
    NMBGF_PutFloat   (hOC, RADIUS);

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutMAPTCommand
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  P)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral    (hOC, "MAPT");
    NMBGF_PutCoordinate (hOC, P);

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutMAPHCommand
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  P,
     NMBGF_FLOAT       HEAD)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral    (hOC, "MAPH");
    NMBGF_PutCoordinate (hOC, P);
    NMBGF_PutFloat      (hOC, HEAD);

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutTNRLCommand
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  ANGLE)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "TNRL");
    NMBGF_PutFloat   (hOC, ANGLE);

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutTNAHCommand
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  HEAD)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "TNAH");
    NMBGF_PutFloat   (hOC, HEAD);

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutTRFPCommand
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  P)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral    (hOC, "TRFP");
    NMBGF_PutCoordinate (hOC, P);

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutTL90Command
    (NMBGF_HOC  hOC)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "TL90");

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutTR90Command
    (NMBGF_HOC  hOC)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "TR90");

    NMBGF_IncrementTopClientStackCounter (hOC);
};



void NMBGF_LINC_PutT180Command
    (NMBGF_HOC  hOC)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutLiteral (hOC, "T180");

    NMBGF_IncrementTopClientStackCounter (hOC);
};


void NMBGF_LINC_CommandsDone
    (NMBGF_HOC  hOC)
{
    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC, NMBGF_GetTopClientStackCounter (hOC));

    NMBGF_PopOutputStreamBuffer (hOC);
};








void NMBGF_PutLINMSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  CATAGORY)
/*
    Writes a LINM section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.

    Instructions for writing an LINM section.
        1. Call NMBGF_PutLINMSection
        2. For every continuous line in the geographic feature...
            2.1 Call NMBGF_LINM_BeginPoints.
            2.2 Call NMBGF_LINM_PutPoint for every point in the continuous line.
            2.3 Call NMBGF_LINM_PointsDone.
        3. Call NMBGF_LINM_ContinuousLinesDone.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "LINM");

    NMBGF_PutString (hOC, CATAGORY);

    NMBGF_PushOutputStreamBuffer (hOC);

    NMBGF_SetTopClientStackCounter (hOC, 0);
};


void NMBGF_LINM_BeginPoints
    (NMBGF_HOC  hOC)
{
    NMBGF_PushOutputStreamBuffer (hOC);
};


void NMBGF_LINM_PutPoint
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  P)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutCoordinate (hOC, P);
};


void NMBGF_LINM_PointsDone
    (NMBGF_HOC hOC)
{
    assert (hOC->StackHeight > -1);


    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC,
         NMBGF_GetTopStackElementCounter (hOC));

    NMBGF_PopOutputStreamBuffer (hOC);

    NMBGF_IncrementTopClientStackCounter (hOC);
};


void NMBGF_LINM_ContinuousLinesDone
    (NMBGF_HOC hOC)
{
    assert (hOC->StackHeight > -1);


    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC, NMBGF_GetTopClientStackCounter (hOC));

    NMBGF_PopOutputStreamBuffer (hOC);
};








void NMBGF_PutLINSSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  CATAGORY)
/*
    Writes a LINS section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.

    Instructions for writing a LINS section.
        1. Call NMBGF_PutLINSSection
        2. Call NMBGF_LINS_PutPoint for every point in the line.
        3. Call NMBGF_LINS_PointsDone.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "LINS");

    NMBGF_PutString (hOC, CATAGORY);

    NMBGF_PushOutputStreamBuffer (hOC);
};


void NMBGF_LINS_PutPoint
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  P)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutCoordinate (hOC, P);
};


void NMBGF_LINS_PointsDone
    (NMBGF_HOC hOC)
{
    assert (hOC->StackHeight > -1);


    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC,
         NMBGF_GetTopStackElementCounter (hOC));

    NMBGF_PopOutputStreamBuffer (hOC);
};








void NMBGF_PutMTRCSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  TYPE,
     NMBGF_STRING  UNIT)
/*
    Writes a MTRC section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "MTRC");

    NMBGF_PutString (hOC, TYPE);
    NMBGF_PutString (hOC, UNIT);
};








void NMBGF_PutPERSSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  NAME,
     NMBGF_STRING  TITLE,
     NMBGF_STRING  COMPANY,
     NMBGF_STRING  ADDRESS,
     NMBGF_STRING  PHONE,
     NMBGF_STRING  FAX,
     NMBGF_STRING  EMAIL)
/*
    Writes a PERS section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "PERS");

    NMBGF_PutString (hOC, NAME);
    NMBGF_PutString (hOC, TITLE);
    NMBGF_PutString (hOC, COMPANY);
    NMBGF_PutString (hOC, ADDRESS);
    NMBGF_PutString (hOC, PHONE);
    NMBGF_PutString (hOC, FAX);
    NMBGF_PutString (hOC, EMAIL);
};








void NMBGF_PutPNTMSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  CATAGORY)
/*
    Writes a PNTM section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.

    Instructions for writing a PNTM section.
        1. Call NMBGF_PutPNTMSection
        2. Call NMBGF_PNTM_PutPoint for every point in the geographic feature.
        3. Call NMBGF_PNTM_PointsDone.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "PNTM");

    NMBGF_PutString (hOC, CATAGORY);

    NMBGF_PushOutputStreamBuffer (hOC);
};


void NMBGF_PNTM_PutPoint
    (NMBGF_HOC         hOC,
     NMBGF_COORDINATE  P)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutCoordinate (hOC, P);
};


void NMBGF_PNTM_PointsDone
    (NMBGF_HOC hOC)
{
    assert (hOC->StackHeight > -1);


    NMBGF_WriteIntegerToPreceedingBuffer
        (hOC,
         NMBGF_GetTopStackElementCounter (hOC));

    NMBGF_PopOutputStreamBuffer (hOC);
};








void NMBGF_PutPNTSSection
    (NMBGF_HOC         hOC,
     NMBGF_STRING      CATAGORY,
     NMBGF_COORDINATE  P)
/*
    Writes a PNTS section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "PNTS");

    NMBGF_PutString     (hOC, CATAGORY);
    NMBGF_PutCoordinate (hOC, P);
};








void NMBGF_PutPROGSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  USERNAME,
     NMBGF_STRING  FILENAME,
     NMBGF_FLOAT   VERSION)
/*
    Writes a PROG section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "PROG");

    NMBGF_PutString (hOC, USERNAME);
    NMBGF_PutString (hOC, FILENAME);
    NMBGF_PutFloat  (hOC, VERSION);
};








void NMBGF_PutSORCSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  CATAGORY)
/*
    Writes a SORC section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "SORC");

    NMBGF_PutString (hOC, CATAGORY);
};








void NMBGF_PutSUBGSection
    (NMBGF_HOC      hOC,
     NMBGF_STRING   NAME,
     NMBGF_STRING   PARENT,
     NMBGF_INTEGER  Iparent,
     NMBGF_INTEGER  Jparent,
     NMBGF_INTEGER  NI,
     NMBGF_INTEGER  NJ)
/*
    Writes a SUBG section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.

    Instructions for writing a SUBG section.
        1. Call NMBGF_PutSUBGSection
        2. For every grid point in the subgrid...
            2.1 Call NMBGF_SUBG_PutGridPoint

    The following code can be used to write the grid points:

        for (i = 0; i < NI; ++i)
            for (j = 0; j < NJ; ++j)
                if (NMBGF_SUBG_IsOdd (i) || NMBGF_SUBG_IsOdd (j))
                    NMBGF_SUBG_PutGridPoint (hOC, V [i][j]);
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "SUBG");

    NMBGF_PutString     (hOC, NAME);
    NMBGF_PutString     (hOC, PARENT);
    NMBGF_PutInteger    (hOC, Iparent);
    NMBGF_PutInteger    (hOC, Jparent);
    NMBGF_PutInteger    (hOC, NI);
    NMBGF_PutInteger    (hOC, NJ);
};


int NMBGF_SUBG_IsOdd (NMBGF_INTEGER  I)
/*
    Returns true if I is an odd number.  This function is provided for
    determining which grid points to output from a subgrid.  See the
    code sample in the comments for function NMBGF_PutSUBGSection.
*/
{
    return (I & 1);
};


void NMBGF_SUBG_PutGridPoint
    (NMBGF_HOC    hOC,
     NMBGF_FLOAT  V)
{
    if (hOC->ASCIIMode)
        NMBGF_PutToTopOutputStreamBuffer_EndOfLine (hOC);

    NMBGF_PutFloat (hOC, V);
};








void NMBGF_PutTIMESection
    (NMBGF_HOC      hOC,
     NMBGF_INTEGER  HOUR,
     NMBGF_INTEGER  MINUTE,
     NMBGF_INTEGER  SECOND)
/*
    Writes a TIME section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "TIME");

    NMBGF_PutInteger (hOC, HOUR);
    NMBGF_PutInteger (hOC, MINUTE);
    NMBGF_PutInteger (hOC, SECOND);
};








void NMBGF_PutTITLSection
    (NMBGF_HOC hOC)
/*
    Writes a TITL section.  This function is not called directly by the
    user.  Rather, it is automatically called by the function that creates
    the NMBGF Output Context (either NMBGF_CreateOutputContext or
    NMBGF_CreateOutputContextAttachedToFile).
*/
{
    if (hOC->ASCIIMode)
        PutToTopOutputStreamBuffer_OpeningBrace (hOC);

    NMBGF_PutLiteral (hOC, "TITL");

    NMBGF_PushOutputStreamBuffer (hOC);

    NMBGF_PutLiteral (hOC, "Grid");
    NMBGF_PutLiteral (hOC, "Vers");
    NMBGF_PutInteger (hOC, MajorNMBGFVersion);
    NMBGF_PutInteger (hOC, MinorNMBGFVersion);
};








void NMBGF_PutUTMCSection
    (NMBGF_HOC     hOC,
     NMBGF_INTEGER ZONE,
     NMBGF_FLOAT   FE,
     NMBGF_FLOAT   FN)
/*
    Writes a UTMC section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "UTMC");

    NMBGF_PutInteger (hOC, ZONE);
    NMBGF_PutFloat   (hOC, FE);
    NMBGF_PutFloat   (hOC, FN);
};








void NMBGF_PutWARNSection
    (NMBGF_HOC     hOC,
     NMBGF_STRING  MESSAGE)
/*
    Writes a WARN section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_PutStandardSectionProlog (hOC, "WARN");

    NMBGF_PutString (hOC, MESSAGE);
};








void NMBGF_PutZCRDSection
    (NMBGF_HOC             hOC,
     NMBGF_FLOAT           Z,
     NMBGF_ZCRD_UNIT_ENUM  UNIT)
/*
    Writes a ZCRD section.  See the document "A Description of the Noise Model
    Binary Grid Format" for a description of the parameters.
*/
{
    NMBGF_LITERAL UnitLiterals [4] = {"FAGL", "MAGL", "FMSL", "MMSL"};

    NMBGF_PutStandardSectionProlog (hOC, "ZCRD");

    NMBGF_PutFloat (hOC, Z);

    NMBGF_PutLiteral (hOC, UnitLiterals [UNIT]);
};








/******************************************************************************/
/*
/*    Functions to Open and Close an NMBGF File
/*
/******************************************************************************/




NMBGF_HOC NMBGF_CreateOutputContext
    (void*                  FileHandle,
     NMBGF_FileCloseFunType FileCloseFun,
     NMBGF_FileWriteFunType FileWriteFun)
{
    return NMBGF_CreateOutputContext2
        (FileHandle,
         FileCloseFun,
         FileWriteFun,
         NMBGF_Options_OutputMode_Binary);
};




NMBGF_HOC NMBGF_CreateOutputContext2
    (void*                  FileHandle,
     NMBGF_FileCloseFunType FileCloseFun,
     NMBGF_FileWriteFunType FileWriteFun,
     unsigned long          Options)
{
    NMBGF_HOC hOC;


    assert (FileHandle);


    hOC = (NMBGF_HOC) malloc (sizeof (struct NMBGF_OutputContext));

    InitializeOutputStreamBufferStack (hOC);

    hOC->WorkBuffer       = NULL;
    hOC->WorkBufferLength = 0;
    hOC->ErrorFlag        = 0;
    hOC->FileHandle       = FileHandle;
    hOC->FileCloseFun     = FileCloseFun;
    hOC->FileWriteFun     = FileWriteFun;
    hOC->LastElement      = LastElement_EndOfLine;
    hOC->IndentLevel      = 0;

    hOC->ASCIIMode = (Options & NMBGF_Options_OutputMode_ASCII) ? 1 : 0;


    NMBGF_PutTITLSection (hOC);

    return hOC;
};




int DefaultFileCloseFun (void* FileHandle)
/*
    Assume that FileHandle actually points to a FILE* created by a call
    to the standard library's fopen function.

    Pass parameters of this call on the the standard library's fclose
    function.

    Return true if error occurs.
*/
{
    return (fclose ((FILE*) FileHandle) != 0);
};


int DefaultFileWriteFun (void* FileHandle, void* Data, NMBGF_INTEGER DataLength)
/*
    Assume that FileHandle actually points to a FILE* created by a call
    to the standard library's fopen function.

    Pass parameters of this call on the the standard library's fwrite
    function.

    Return true if error occurs.
*/
{
    return ((NMBGF_INTEGER) fwrite
                (Data, (size_t) 1, (size_t) DataLength, (FILE*) FileHandle)
            != DataLength);
};




NMBGF_HOC NMBGF_CreateOutputContextAttachedToFile (const char* FileName)
{
    return NMBGF_CreateOutputContextAttachedToFile2
        (FileName, NMBGF_Options_OutputMode_Binary);
};




NMBGF_HOC NMBGF_CreateOutputContextAttachedToFile2
    (const char* FileName, unsigned long Options)
{
    FILE* fot = fopen (FileName, "wb");

    if (fot == NULL)
        return NULL;

    return NMBGF_CreateOutputContext2
        ((void*) fot, DefaultFileCloseFun, DefaultFileWriteFun, Options);
};








void NMBGF_EndWritingOutputFile (NMBGF_HOC hOC)
{
    assert (hOC->FileHandle != NULL);


    NMBGF_PutENDFSection (hOC);


    if (hOC->FileCloseFun (hOC->FileHandle))
        SetErrorFlag (hOC);

    hOC->FileHandle = NULL;


    DeinitializeOutputStreamBufferStack (hOC);

    free (hOC->WorkBuffer);

    free (hOC);
};








int NMBGF_IsInASCIIMode (NMBGF_HOC hOC)
/*
    Returns 1 if the NMBGF output context handle hOC refers to a an
    NMAGF ASCII grid file.  Returns 0 if it refers to an NMBGF binary
    grid file.
*/
{
    return hOC->ASCIIMode;
};



