
/*
    This file demonstrates the use of the Noise Model Binary Grid File (NMBGF)
    I/O Library.  This program creates an NMBGF file named "samplecp.grd".

    The writing of most NMGF sections is demonstrated.

    This program is identical to sample.c, except that it uses the C++
    interface to the NMBGF I/O Library.
*/




/*  Include the NMBGF I/O Library C++ declarations  */

#ifndef nmbgfioc_included
    #include "nmbgfioc.h"
#endif




NMBGF_FLOAT ComputeSampleGridPointValue (NMBGF_COORDINATE C)
/*                                           */
/*  This function calculates values for our  */
/*  grids points as a function of location.  */
/*                                           */
{
    return 100 - (C.C1*C.C1 + C.C2*C.C2)/40000;
};




void WriteGridFile (char* FileName, unsigned long Options)
{
    NMBGF_COORDINATE C;
    NMBGF_INTEGER    i, j;
    NMBGF_FLOAT      V;


    /*  Create a NMGF file named FileName.                        */
    /*  Options will control whether a NMBGF Binary Subtype grid  */
    /*  file or a NMAGF ASCII Subtype grid file is generated.     */
    /*  The TITL section is written automatically.                */

    NMBGF_OutputTranslator OT (FileName, Options);


    /*   Define a Cartesian coordinate system with the CART section.  */

    OT.PutCARTSection (-90, 45, 0, 0, NMBGF_CART_Feet, 0);


    /*  Write a SORC section to specify the source of this NMBGF file.  */

    OT.PutSORCSection ("Demo");


    /*  The SORC will have some optional subsections  */

    OT.BeginOptionalSubsections ();


        /*  Specify long and short descriptions of the NMBGF file.  */

        OT.PutDESLSection
            ("This is a sample NMBGF file.  It contains fake data.  It was "
             "created to illustrate the use of the NMBGF I/O Library.");

        OT.PutDESSSection ("Sample NMBGF file");


        /*  Specify person who ran this program and created the NMBGF file.  */

        OT.PutPERSSection
            ("John Smith",
             "Programmer",
             "Acme Corp.",
             "123 Acme Drive, Champaign, IL 61821",
             "(217) 345-1234",
             "(217) 345-5678",
             "john@acme.com");


        /*  Specify program that created the NMBGF file.  */

        OT.PutPROGSection
            ("NMBGF I/O Library Sample Program (C++ Version)",
             "samplecp.exe",
             1.00);


        /*  Specify when we ran this program and created the NMBGF file.  */

        OT.PutTIMESection (10, 30,   20);
        OT.PutDATESection ( 1,  1, 1996);


    /*  No more optional subsections for the SORC section.  */

    OT.EndOptionalSubsections ();




    /*  Write some geographic attributes to the file.  */



    /*  Write a single-point attribute.  */

    C.C1 = 100; C.C2 = 200;

    OT.PutPNTSSection ("School", C);


    /*  Supply some additional information about  */
    /*  the school with optional subsections.     */

    OT.BeginOptionalSubsections ();

        /*  Specify a short description of the school.  */

        OT.PutDESSSection ("Washington Elementary School");


        /*  Specify the altitude of the school.  */

        OT.PutZCRDSection (500, NMBGF_ZCRD_MetersAboveMeanSeaLevel);


        /*  Specify that the school faces east.  */

        OT.PutHEADSection (90);


        /*  Specify how many students attend this school.  */

        OT.PutATRISection ("Students", 350);


        /*  Specify the principal's name.  */

        OT.PutATRSSection ("Principal", "Jane Doe");


        /*  Specify a table that gives measured background noise  */
        /*  levels as a function of time and data.                */

        OT.PutATRTSection ("Measured Background Noise");


            /*  Specify the columns of the table.  */

                OT.ATRT_AddColumn ("Time",        NMBGF_ATRT_String);
                OT.ATRT_AddColumn ("Date",        NMBGF_ATRT_String);
                OT.ATRT_AddColumn ("Noise Level", NMBGF_ATRT_Float);

            OT.ATRT_ColumnsDone ();


            /*  Specify each row of the table.  */

                OT.ATRT_PutString ("4:00 AM");
                OT.ATRT_PutString ("22 Aug 1996");
                OT.ATRT_PutFloat  (42.3);

            OT.ATRT_RowDone ();

                OT.ATRT_PutString ("9:00 AM");
                OT.ATRT_PutString ("22 Aug 1996");
                OT.ATRT_PutFloat  (54.2);

            OT.ATRT_RowDone ();

                OT.ATRT_PutString ("2:00 AM");
                OT.ATRT_PutString ("23 Aug 1996");
                OT.ATRT_PutFloat  (41.9);

            OT.ATRT_RowDone ();


        /*  No more rows: we are done with this table.  */

        OT.ATRT_TableDone ();


    /*  No more optional subsections for the school.  */

    OT.EndOptionalSubsections ();



    /*  Write a multiple-point attribute that specifies  */
    /*  the locations of all nearby hospitals.           */

    OT.PutPNTMSection ("Hospital");

        C.C1 = 500; C.C2 = 800;
        OT.PNTM_PutPoint (C);

        C.C1 = -500; C.C2 = 300;
        OT.PNTM_PutPoint (C);

        C.C1 = 200; C.C2 = -250;
        OT.PNTM_PutPoint (C);

    OT.PNTM_PointsDone ();



    /*  Write a line geographic attribute that represents a major road.  */

    C.C1 = -1000; C.C2 = -1000;

    OT.PutLINCSection
        ("Primary Road",
         NMBGF_LINC_Feet,
         C,
         45);

    OT.LINC_PutDSTRCommand (700);
    OT.LINC_PutDARRCommand (-45, 200);
    OT.LINC_PutDSTRCommand (1000);

    OT.LINC_CommandsDone ();

    /*  Supply some additional information about  */
    /*  the road with optional subsections.       */

    OT.BeginOptionalSubsections ();

        OT.PutDESSSection ("US Highway 66");

        OT.PutWARNSection
            ("Based on old, possibly innacurate, survey data.");

    OT.EndOptionalSubsections ();



    /*  Write a line geographic feature that represents other roads.  */

    OT.PutLINMSection ("Secondary Road");

        OT.LINM_BeginPoints ();

            C.C1 = -100; C.C2 = -100;
            OT.LINM_PutPoint (C);

            C.C1 = -100; C.C2 = 100;
            OT.LINM_PutPoint (C);

            C.C1 = -300; C.C2 = -400;
            OT.LINM_PutPoint (C);

        OT.LINM_PointsDone ();

        OT.LINM_BeginPoints ();

            C.C1 = 500; C.C2 = -300;
            OT.LINM_PutPoint (C);

            C.C1 = 600; C.C2 = 800;
            OT.LINM_PutPoint (C);

        OT.LINM_PointsDone ();

    OT.LINM_ContinuousLinesDone ();



    /*  Write a line geographic feature that represents a railroad track.  */

    OT.PutLINSSection ("Railroad");

        C.C1 = -700; C.C2 = 400;
        OT.LINS_PutPoint (C);

        C.C1 = 800; C.C2 = 500;
        OT.LINS_PutPoint (C);

        C.C1 =  900; C.C2 = -900;
        OT.LINS_PutPoint (C);

    OT.LINS_PointsDone ();



    /*  Write an area geographic feature that represents urban areas.  */

    OT.PutAREMSection ("Urban Area");

        OT.AREM_BeginPoints ();

            C.C1 =  -200; C.C2 = -200;
            OT.AREM_PutPoint (C);

            C.C1 =  200; C.C2 = -200;
            OT.AREM_PutPoint (C);

            C.C1 =  200; C.C2 =  200;
            OT.AREM_PutPoint (C);

            C.C1 = -200; C.C2 =  200;
            OT.AREM_PutPoint (C);

        OT.AREM_PointsDone ();

        OT.AREM_BeginPoints ();

            C.C1 =  500; C.C2 = 500;
            OT.AREM_PutPoint (C);

            C.C1 =  950; C.C2 = 850;
            OT.AREM_PutPoint (C);

            C.C1 =  900; C.C2 = 450;
            OT.AREM_PutPoint (C);

        OT.AREM_PointsDone ();

    OT.AREM_ClosedPolygonsDone ();



    /*  Write an area geographic feature that represents a lake.  */

    OT.PutARESSection ("Water");

        C.C1 =  -400; C.C2 = -550;
        OT.ARES_PutPoint (C);

        C.C1 =  -500; C.C2 = -600;
        OT.ARES_PutPoint (C);

        C.C1 =  -600; C.C2 = -350;
        OT.ARES_PutPoint (C);

    OT.ARES_PointsDone ();

    OT.BeginOptionalSubsections ();

        OT.PutDESSSection ("Lake Mead");

    OT.EndOptionalSubsections ();



    /*  Write a Background Map section that shows local roads.  */

    OT.PutBKMPSection ("c:\\mapdata\\roads.bmp", "Primary Road");



    /*  Now specify a grid of points representing modeled noise levels  */


    /*  Specify the metric.  */

    OT.PutMTRCSection ("Noise", "DNL");


    /*  Ignore grid points with a value below -40  */

    OT.PutGTSHSection (-40, 9.9e9);


    /*  Specify the area where the noise levels are known.  */

    OT.PutDAPYSection ();

        OT.DAPY_BeginPoints ();

            C.C1 =  -1000; C.C2 = -1000;
            OT.DAPY_PutPoint (C);

            C.C1 =   1000; C.C2 = -1000;
            OT.DAPY_PutPoint (C);

            C.C1 =   1000; C.C2 =  1000;
            OT.DAPY_PutPoint (C);

            C.C1 =  -1000; C.C2 =  1000;
            OT.DAPY_PutPoint (C);

            C.C1 =  -1000; C.C2 = -1000;
            OT.DAPY_PutPoint (C);

        OT.DAPY_PointsDone ();

    OT.DAPY_ClosedPolygonsDone ();


    /*  Write a rectangular primary grid.  */

    C.C1 =  -1000; C.C2 = -1000;
    OT.PutGRIDSection
        ("Grid 1",
         11,
         11,
         200,
         200,
         NMBGF_GRID_Feet,
         C,
         0);

    for (i = 0; i < 11; ++i)
        for (j = 0; j < 11; ++j)
        {
            C.C1 = -1000 + i * 200;  /*  Compute X-Y coordinates  */
            C.C2 = -1000 + j * 200;  /*  of this grid point.      */

            V = ComputeSampleGridPointValue (C);

            OT.GRID_PutGridPoint (V);
                              /*  ^ Obviously,  in a real model, we  */
                              /*  would be writing real data here.   */
        };


    /*  Write a nested subgrid that provides additional  */
    /*  grid points to increase the resolution.          */

    OT.PutSUBGSection
        ("Subgrid 1",
         "Grid 1",
         4,
         3,
         5,
         5);

    for (i = 0; i < 5; ++i)
        for (j = 0; j < 5; ++j)
            if (NMBGF_SUBG_IsOdd (i) || NMBGF_SUBG_IsOdd (j))
            {
                C.C1 = -400 + i * 100;  /*  Compute X-Y coordinates  */
                C.C2 = -600 + j * 100;  /*  of this subgrid point.   */

                V = ComputeSampleGridPointValue (C);

                OT.SUBG_PutGridPoint (V);
                                  /*  ^ In a real model, we would  */
                                  /*  be writing real data here.   */
            };



    /*  We have a few more points that aren't arranged on a   */
    /*  rectangular grid.  Write them using a DPAL section.   */

    OT.PutDPALSection ();

        for (i = 0; i < 5; ++i)
        {
            C.C1 = -900; C.C2 = -900 + i*50.0;

            V = ComputeSampleGridPointValue (C);

            OT.DPAL_PutDataPoint (C, V);
                              /*  ^  ^ In a real model, we would be  */
                              /*       writing real coordinates and  */
                              /*       data here.       */
        };

    OT.DPAL_DataPointsDone ();


    /*  Demonstrate how to write a custom section         */
    /*  that is not part of the NMBGF standard.           */
    /*  We will name our hypothetical section 'XXXX',     */
    /*  and define it to have a single string parameter.  */

    OT.PutStandardSectionProlog ("XXXX");

    OT.PutString ("I am an example unrecornized NMBGF section.");


    /*  We are finished.  The destructor for OT will automatically  */
    /*  write the ENDF section and close the NMBGF file.            */
};




int main ()
{
    /*  Write two files with the same information.  The first will  */
    /*  be a NMBGF Binary Subtype file, and the second will be a    */
    /*  NMAGF ASCII Subtype file.                                   */

    WriteGridFile ("bpsample.grd", NMBGF_Options_OutputMode_Binary);
    WriteGridFile ("apsample.grd", NMBGF_Options_OutputMode_ASCII);

    return 0;
};


