
/*
    This file demonstrates the use of the Noise Model Grid File (NMGF)
    I/O Library.  This program creates an NMBGF Binary Subtype file named
    "bsample.grd" and a NMAGF ASCII Subtype file names "asample.grd".

    The writing of most NMGF sections is demonstrated.
*/




/*  Include the NMBGF I/O Library declarations  */

#ifndef nmbgfiol_included
    #include "nmbgfiol.h"
#endif




NMBGF_FLOAT ComputeSampleGridPointValue (NMBGF_COORDINATE C)
/*                                           */
/*  This function calculates values for our  */
/*  grids points as a function of location.  */
/*                                           */
{
    return 100 - (C.C1*C.C1 + C.C2*C.C2)/40000;
};




void WriteGridFile (char* FileName, unsigned long Options)
{
    NMBGF_COORDINATE C;
    NMBGF_HOC        hOC;
    NMBGF_INTEGER    i, j;
    NMBGF_FLOAT      V;


    /*  Create a NMGF file named FileName.                        */
    /*  Options will control whether a NMBGF Binary Subtype grid  */
    /*  file or a NMAGF ASCII Subtype grid file is generated.     */
    /*  The TITL section is written automatically.                */

    hOC = NMBGF_CreateOutputContextAttachedToFile2 (FileName, Options);


    /*   Define a Cartesian coordinate system with the CART section.  */

    NMBGF_PutCARTSection (hOC, -90, 45, 0, 0, NMBGF_CART_Feet, 0);


    /*  Write a SORC section to specify the source of this NMBGF file.  */

    NMBGF_PutSORCSection (hOC, "Demonstration");


    /*  The SORC will have some optional subsections  */

    NMBGF_BeginOptionalSubsections (hOC);


        /*  Specify long and short descriptions of the NMBGF file.  */

        NMBGF_PutDESLSection
            (hOC,
             "This is a sample NMBGF file.  It contains fake data.  It was "
             "created to illustrate the use of the NMBGF I/O Library.");

        NMBGF_PutDESSSection (hOC, "Sample NMBGF file");


        /*  Specify person who ran this program and created the NMBGF file.  */

        NMBGF_PutPERSSection
            (hOC,
             "John Smith",
             "Programmer",
             "Acme Corp.",
             "123 Acme Drive, Champaign, IL 61821",
             "(217) 345-1234",
             "(217) 345-5678",
             "john@acme.com");


        /*  Specify program that created the NMBGF file.  */

        NMBGF_PutPROGSection
            (hOC,
             "NMBGF I/O Library Sample Program",
             "sample.exe",
             1.00);


        /*  Specify when we ran this program and created the NMBGF file.  */

        NMBGF_PutTIMESection (hOC, 10, 30, 20);
        NMBGF_PutDATESection (hOC,  1,  1, 1996);


    /*  No more optional subsections for the SORC section.  */

    NMBGF_EndOptionalSubsections (hOC);




    /*  Write some geographic attributes to the file.  */



    /*  Write a single-point attribute.  */

    C.C1 = 100; C.C2 = 200;

    NMBGF_PutPNTSSection (hOC, "School", C);


    /*  Supply some additional information about  */
    /*  the school with optional subsections.     */

    NMBGF_BeginOptionalSubsections (hOC);

        /*  Specify a short description of the school.  */

        NMBGF_PutDESSSection (hOC, "Washington Elementary School");


        /*  Specify the altitude of the school.  */

        NMBGF_PutZCRDSection (hOC, 500, NMBGF_ZCRD_MetersAboveMeanSeaLevel);


        /*  Specify that the school faces east.  */

        NMBGF_PutHEADSection (hOC, 90);


        /*  Specify how many students attend this school.  */

        NMBGF_PutATRISection (hOC, "Students", 350);


        /*  Specify the principal's name.  */

        NMBGF_PutATRSSection (hOC, "Principal", "Jane Doe");


        /*  Specify a table that gives measured background noise  */
        /*  levels as a function of time and data.                */

        NMBGF_PutATRTSection (hOC, "Measured Background Noise");


            /*  Specify the columns of the table.  */

                NMBGF_ATRT_AddColumn (hOC, "Time",        NMBGF_ATRT_String);
                NMBGF_ATRT_AddColumn (hOC, "Date",        NMBGF_ATRT_String);
                NMBGF_ATRT_AddColumn (hOC, "Noise Level", NMBGF_ATRT_Float);

            NMBGF_ATRT_ColumnsDone (hOC);


            /*  Specify each row of the table.  */

                NMBGF_ATRT_PutString (hOC, "4:00 AM");
                NMBGF_ATRT_PutString (hOC, "22 Aug 1996");
                NMBGF_ATRT_PutFloat  (hOC, 42.3);

            NMBGF_ATRT_RowDone (hOC);

                NMBGF_ATRT_PutString (hOC, "9:00 AM");
                NMBGF_ATRT_PutString (hOC, "22 Aug 1996");
                NMBGF_ATRT_PutFloat  (hOC, 54.2);

            NMBGF_ATRT_RowDone (hOC);

                NMBGF_ATRT_PutString (hOC, "2:00 AM");
                NMBGF_ATRT_PutString (hOC, "23 Aug 1996");
                NMBGF_ATRT_PutFloat  (hOC, 41.9);

            NMBGF_ATRT_RowDone (hOC);


        /*  No more rows: we are done with this table.  */

        NMBGF_ATRT_TableDone (hOC);


    /*  No more optional subsections for the school.  */

    NMBGF_EndOptionalSubsections (hOC);



    /*  Write a multiple-point attribute that specifies  */
    /*  the locations of all nearby hospitals.           */

    NMBGF_PutPNTMSection (hOC, "Hospital");

        C.C1 = 500; C.C2 = 800;
        NMBGF_PNTM_PutPoint (hOC, C);

        C.C1 = -500; C.C2 = 300;
        NMBGF_PNTM_PutPoint (hOC, C);

        C.C1 = 200; C.C2 = -250;
        NMBGF_PNTM_PutPoint (hOC, C);

    NMBGF_PNTM_PointsDone (hOC);



    /*  Write a line geographic attribute that represents a major road.  */

    C.C1 = -1000; C.C2 = -1000;

    NMBGF_PutLINCSection
        (hOC,
         "Primary Road",
         NMBGF_LINC_Feet,
         C,
         45);

    NMBGF_LINC_PutDSTRCommand (hOC, 700);
    NMBGF_LINC_PutDARRCommand (hOC, -45, 200);
    NMBGF_LINC_PutDSTRCommand (hOC, 1000);

    NMBGF_LINC_CommandsDone (hOC);

    /*  Supply some additional information about  */
    /*  the road with optional subsections.       */

    NMBGF_BeginOptionalSubsections (hOC);

        NMBGF_PutDESSSection (hOC, "US Highway 66");

        NMBGF_PutWARNSection
            (hOC, "Based on old, possibly innacurate, survey data.");

    NMBGF_EndOptionalSubsections (hOC);



    /*  Write a line geographic feature that represents other roads.  */

    NMBGF_PutLINMSection (hOC, "Secondary Road");

        NMBGF_LINM_BeginPoints (hOC);

            C.C1 = -100; C.C2 = -100;
            NMBGF_LINM_PutPoint (hOC, C);

            C.C1 = -100; C.C2 = 100;
            NMBGF_LINM_PutPoint (hOC, C);

            C.C1 = -300; C.C2 = -400;
            NMBGF_LINM_PutPoint (hOC, C);

        NMBGF_LINM_PointsDone (hOC);

        NMBGF_LINM_BeginPoints (hOC);

            C.C1 = 500; C.C2 = -300;
            NMBGF_LINM_PutPoint (hOC, C);

            C.C1 = 600; C.C2 = 800;
            NMBGF_LINM_PutPoint (hOC, C);

        NMBGF_LINM_PointsDone (hOC);

    NMBGF_LINM_ContinuousLinesDone (hOC);



    /*  Write a line geographic feature that represents a railroad track.  */

    NMBGF_PutLINSSection (hOC, "Railroad");

        C.C1 = -700; C.C2 = 400;
        NMBGF_LINS_PutPoint (hOC, C);

        C.C1 = 800; C.C2 = 500;
        NMBGF_LINS_PutPoint (hOC, C);

        C.C1 =  900; C.C2 = -900;
        NMBGF_LINS_PutPoint (hOC, C);

    NMBGF_LINS_PointsDone (hOC);



    /*  Write an area geographic feature that represents urban areas.  */

    NMBGF_PutAREMSection (hOC, "Urban Area");

        NMBGF_AREM_BeginPoints (hOC);

            C.C1 =  -200; C.C2 = -200;
            NMBGF_AREM_PutPoint (hOC, C);

            C.C1 =  200; C.C2 = -200;
            NMBGF_AREM_PutPoint (hOC, C);

            C.C1 =  200; C.C2 =  200;
            NMBGF_AREM_PutPoint (hOC, C);

            C.C1 = -200; C.C2 =  200;
            NMBGF_AREM_PutPoint (hOC, C);

        NMBGF_AREM_PointsDone (hOC);

        NMBGF_AREM_BeginPoints (hOC);

            C.C1 =  500; C.C2 = 500;
            NMBGF_AREM_PutPoint (hOC, C);

            C.C1 =  950; C.C2 = 850;
            NMBGF_AREM_PutPoint (hOC, C);

            C.C1 =  900; C.C2 = 450;
            NMBGF_AREM_PutPoint (hOC, C);

        NMBGF_AREM_PointsDone (hOC);

    NMBGF_AREM_ClosedPolygonsDone (hOC);



    /*  Write an area geographic feature that represents a lake.  */

    NMBGF_PutARESSection (hOC, "Water");

        C.C1 =  -400; C.C2 = -550;
        NMBGF_ARES_PutPoint (hOC, C);

        C.C1 =  -500; C.C2 = -600;
        NMBGF_ARES_PutPoint (hOC, C);

        C.C1 =  -600; C.C2 = -350;
        NMBGF_ARES_PutPoint (hOC, C);

    NMBGF_ARES_PointsDone (hOC);

    NMBGF_BeginOptionalSubsections (hOC);

        NMBGF_PutDESSSection (hOC, "Lake Mead");

    NMBGF_EndOptionalSubsections (hOC);



    /*  Write a Background Map section that shows local roads.  */

    NMBGF_PutBKMPSection (hOC, "c:\\mapdata\\roads.bmp", "Primary Road");



    /*  Now specify a grid of points representing modeled noise levels  */


    /*  Specify the metric.  */

    NMBGF_PutMTRCSection (hOC, "Noise", "DNL");


    /*  Ignore grid points with a value below -40  */

    NMBGF_PutGTSHSection (hOC, -40, 9.9e9);


    /*  Specify the area where the noise levels are known.  */

    NMBGF_PutDAPYSection (hOC);

        NMBGF_DAPY_BeginPoints (hOC);

            C.C1 =  -1000; C.C2 = -1000;
            NMBGF_DAPY_PutPoint (hOC, C);

            C.C1 =   1000; C.C2 = -1000;
            NMBGF_DAPY_PutPoint (hOC, C);

            C.C1 =   1000; C.C2 =  1000;
            NMBGF_DAPY_PutPoint (hOC, C);

            C.C1 =  -1000; C.C2 =  1000;
            NMBGF_DAPY_PutPoint (hOC, C);

            C.C1 =  -1000; C.C2 = -1000;
            NMBGF_DAPY_PutPoint (hOC, C);

        NMBGF_DAPY_PointsDone (hOC);

    NMBGF_DAPY_ClosedPolygonsDone (hOC);


    /*  Write a rectangular primary grid.  */

    C.C1 =  -1000; C.C2 = -1000;
    NMBGF_PutGRIDSection
        (hOC,
         "Grid 1",
         11,
         11,
         200,
         200,
         NMBGF_GRID_Feet,
         C,
         0);

    for (i = 0; i < 11; ++i)
        for (j = 0; j < 11; ++j)
        {
            C.C1 = -1000 + i * 200;  /*  Compute X-Y coordinates  */
            C.C2 = -1000 + j * 200;  /*  of this grid point.      */

            V = ComputeSampleGridPointValue (C);

            NMBGF_GRID_PutGridPoint (hOC, V);
                                      /*  ^ Obviously,  in a real model, we   */
                                      /*    would be writing real data here.  */
        };


    /*  Write a nested subgrid that provides additional  */
    /*  grid points to increase the resolution.          */

    NMBGF_PutSUBGSection
        (hOC,
         "Subgrid 1",
         "Grid 1",
         4,
         3,
         5,
         5);

    for (i = 0; i < 5; ++i)
        for (j = 0; j < 5; ++j)
            if (NMBGF_SUBG_IsOdd (i) || NMBGF_SUBG_IsOdd (j))
            {
                C.C1 = -400 + i * 100;  /*  Compute X-Y coordinates  */
                C.C2 = -600 + j * 100;  /*  of this subgrid point.   */

                V = ComputeSampleGridPointValue (C);

                NMBGF_SUBG_PutGridPoint (hOC, V);
                                          /*  ^ In a real model, we would   */
                                          /*    be writing real data here.  */
            };


    /*  We have a few more points that aren't arranged on a   */
    /*  rectangular grid.  Write them using a DPAL section.   */

    NMBGF_PutDPALSection (hOC);

        for (i = 0; i < 5; ++i)
        {
            C.C1 = -900; C.C2 = -900 + i*50.0;

            V = ComputeSampleGridPointValue (C);

            NMBGF_DPAL_PutDataPoint (hOC, C, V);
                                      /*  ^  ^ In a real model, we would be  */
                                      /*       writing real coordinates and  */
                                      /*       data here.       */
        };

    NMBGF_DPAL_DataPointsDone (hOC);


    /*  Demonstrate how to write a custom section         */
    /*  that is not part of the NMBGF standard.           */
    /*  We will name our hypothetical section 'XXXX',     */
    /*  and define it to have a single string parameter.  */

    NMBGF_PutStandardSectionProlog (hOC, "XXXX");

    NMBGF_PutString (hOC, "I am an example unrecognized NMBGF section.");


    /*  We are finished.  Close the NMBGF file.  The ENDF  */
    /*  section will be automatically written.             */

    NMBGF_EndWritingOutputFile (hOC);
};




int main ()
{
    /*  Write two files with the same information.  The first will  */
    /*  be a NMBGF Binary Subtype file, and the second will be a    */
    /*  NMAGF ASCII Subtype file.                                   */

    WriteGridFile ("bsample.grd", NMBGF_Options_OutputMode_Binary);
    WriteGridFile ("asample.grd", NMBGF_Options_OutputMode_ASCII);

    return 0;
};


